/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/* This MX kernel lib code was originally contributed by
 * Brice.Goglin@ens-lyon.org (LIP/INRIA/ENS-Lyon) */

#ifndef _mx_arch_klib_h_
#define _mx_arch_klib_h_

#include "mx_arch.h"

/* kernel library symbols */

#define mx_klib_symbol(s) EXPORT_SYMBOL(s)

/* kernel library wrappers */

#define mx_klib_sprintf(args...) sprintf(args)
#define mx_klib_printf(args...) printk(args)
#define mx_klib_snprintf(args...) snprintf(args)
#define mx_klib_strlen(s) strlen(s)
#define mx_klib_strtol simple_strtol

static inline int
mx_klib_gettimeofday(struct timeval *tv, struct timezone *tz) {
  if (tv != NULL) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    jiffies_to_timeval(jiffies, tv);
#else
    do_gettimeofday(tv);
#endif
  }
  return 0;
}

/* klib mutex */

typedef struct semaphore mx_klib_mutex_t;

#define mx_klib_mutex_init(mutexp) sema_init(mutexp,1)
#define mx_klib_mutex_destroy(mutexp) do { /* nothing */ } while (0)
#define mx_klib_mutex_lock(mutexp) down(mutexp)
#define mx_klib_mutex_unlock(mutexp) up(mutexp)

/* klib thread */

typedef struct completion mx_klib_thread_t;

static inline
int
mx_klib_thread_create(mx_klib_thread_t *threadp,
		      int (*func)(void *), void *arg) {
  int ret;
  init_completion(threadp);
  ret = kernel_thread(func, arg, CLONE_FS|CLONE_FILES);
  return ret >= 0 ? 0 : -ret;
}

#define mx_klib_thread_return_t int
#define mx_klib_thread_join(threadp) wait_for_completion(threadp)
#define mx_klib_thread_exit(threadp) complete_and_exit(threadp, 0)
#define mx_klib_thread_init(threadp, name) mx_lxx_daemonize(name)

/* copy routines */

int mx_copy_from_user_mm(char *kdst, mx_uaddr_t usrc, struct mm_struct *src_mm, uint32_t len);
int mx_copy_to_user_mm(mx_uaddr_t udst, char *ksrc, struct mm_struct *dst_mm,  uint32_t len);

static inline int
mx_arch_memcpy_from_segment(void *ptr, uint64_t segment_ptr,
			    size_t amount, uintptr_t memory_context)
{
  int status = 0;
  struct mm_struct *mm = (struct mm_struct *)memory_context;
  switch (MX_PIN_CTX_TO_LOWLEVEL(memory_context)) {
  case MX_PIN_PHYSICAL:
    {
      struct page * page;
      unsigned long offset = segment_ptr & (PAGE_SIZE-1);
      uint64_t phys = segment_ptr;
      unsigned long remaining = amount;
      unsigned long to_copy;
      void *vaddr;
      while (remaining) {
	to_copy = PAGE_SIZE - offset;
	if (to_copy > remaining)
	  to_copy = remaining;
	page = mx_linux_phys_to_page(phys);
	vaddr = kmap_atomic(page, KM_USER0);
	if (!vaddr) {
	  MX_WARN(("mx_arch_memcpy_from_segment: kmap_atomic failed !\n"));
	  status = ENOMEM;
	  goto exit;
	}
	memcpy(ptr, vaddr+offset, to_copy);
	kunmap_atomic(vaddr, KM_USER0);
	remaining -= to_copy;
	ptr += to_copy;
	phys += to_copy;
	offset = 0;
      }
    }
    break;
  default:
    if (!MX_PIN_CTX_IS_USER_CTX(memory_context) || mm == current->mm) {
      /* either kernel memory or user memory in the current process, a simple
       * use a simple copy or copyin in the caller.
       */
      return EINVAL;
    } else {
      /* copy from another process user-space */
      status = mx_copy_from_user_mm(ptr, segment_ptr, (void *) memory_context, amount);
    }
    break;
  }
 exit:
  return status;
}

static inline int
mx_arch_memcpy_to_segment(uint64_t segment_ptr, void *ptr,
			  size_t amount, uintptr_t memory_context)
{ 
  int status = 0;
  struct mm_struct *mm = (struct mm_struct *)memory_context;
  switch (MX_PIN_CTX_TO_LOWLEVEL(memory_context)) {
  case MX_PIN_PHYSICAL:
    {
      struct page * page;
      unsigned long offset = segment_ptr & (PAGE_SIZE-1);
      uint64_t phys = segment_ptr;
      unsigned long remaining = amount;
      unsigned long to_copy;
      void *vaddr;
      while (remaining) {
	to_copy = PAGE_SIZE - offset;
	if (to_copy > remaining)
	  to_copy = remaining;
	page = mx_linux_phys_to_page(phys);
	vaddr = kmap_atomic(page, KM_USER0);
	if (!vaddr) {
	  MX_WARN(("mx_arch_memcpy_to_segment: kmap_atomic failed !\n"));
	  status = ENOMEM;
	  goto exit;
	}
	memcpy(vaddr+offset, ptr, to_copy);
	kunmap_atomic(vaddr, KM_USER0);
	remaining -= to_copy;
	ptr += to_copy;
	phys += to_copy;
	offset = 0;
      }
    }
    break;
  default:
    if (!MX_PIN_CTX_IS_USER_CTX(memory_context) || mm == current->mm) {
      /* either kernel memory or user memory in the current process
       * use a simple copy or copyout in the caller.
       */
      return EINVAL;
    } else {
      /* copy to another process user-space */
      status = mx_copy_to_user_mm(segment_ptr, ptr, (void *) memory_context, amount);
    }
    break;
  }
 exit:
  return status;
}

#define MX_UNEXP_QUEUE_LENGTH_MAX (256*PAGE_SIZE)

#endif /* _mx_arch_klib_h_ */
